class OneDisplayDebugger {
  constructor() {
    this.deviceID = getDeviceID();
    this.debugging = false;
    this.debugBarElement = null;
    this.debugProgressElement = null;
    this.debugStartTime = 0;
    this.debugDuration = 0;
    this.debugAnimationFrame = null;
    this.numPlaybackAreas = 1;
  }


  /**
   * Collects and sends diagnostics data.
   */
  async startDebugger() {
    if(!this.debugging) { 
      this.debugging = true;
      this.debugBarElement = document.createElement("div");
      this.debugBarElement.id = "debugBar";
      this.debugBarElement.innerHTML = "";
      this.debugProgressElement = document.createElement("div");
      this.debugProgressElement.id = "debugProgress";
      this.debugBarElement.appendChild(this.debugProgressElement);
      this.debugBarElement.appendChild(document.createElement("br"));
  
      let pairingCode = document.createElement("div");
      pairingCode.id = "pairingCode";
      pairingCode.innerHTML = "Pairing: " + this.deviceID;
  
      let ramUsage = document.createElement("div");
      ramUsage.id = "debug-ram";
      ramUsage.innerHTML = "";
      this.debugBarElement.appendChild(ramUsage);
  
      let diskUsage = document.createElement("div");
      diskUsage.id = "debug-disk";
      diskUsage.innerHTML = "";
      this.debugBarElement.appendChild(diskUsage);
  
      let version = document.createElement("div");
      version.id = "debug-version";
      version.innerHTML = "";
      this.debugBarElement.appendChild(version);
  
      this.debugBarElement.appendChild(pairingCode);
      document.body.appendChild(this.debugBarElement);
      if (typeof window.diagnosticsObject !== 'undefined' && typeof window.diagnosticsObject.showDiagnostics == 'function') {
        window.diagnosticsObject.showDiagnostics();
      }
    }
    

  }

  async stopDebugger() {
    if(this.debugging) {
      this.debugging = false;
      if (this.debugBarElement) {
        document.body.removeChild(this.debugBarElement);
        this.debugBarElement = null;
      }
    }
  }

  async add(message) {
    if (this.debugging && this.debugBarElement) {
      const timeStr = new Date().toLocaleTimeString();
      let node = document.createElement("p");
      node.className = "debug-message";
      node.innerHTML = timeStr + " > " + message;
      let firstChild = document.querySelector("#debugBar p:first-of-type");
      this.debugBarElement.insertBefore(node, firstChild);
      const debugMessages = Array.from(this.debugBarElement.querySelectorAll("p.debug-message"));
      if (debugMessages.length > 30) {
        for (let i = 30; i < debugMessages.length; i++) {
          debugMessages[i].remove();
        }
      }
    }
  }

  nowPlaying(item) {
    if (!this.debugBarElement || !item || this.numPlaybackAreas > 1) return;
    let nowPlayingSpan = this.debugBarElement.querySelector("span.now-playing");
    if (!nowPlayingSpan) {
      nowPlayingSpan = document.createElement("span");
      nowPlayingSpan.className = "now-playing";
      this.debugBarElement.prepend(nowPlayingSpan);
    }
    nowPlayingSpan.innerHTML =
      "Now playing: " +
      item.library_type.toUpperCase() +
      " - " +
      item.name +
      ", duration: " +
      parseFloat(item.show_seconds).toFixed(2) +
      "s)";
    this.startPlaybackIndicator(parseFloat(item.show_seconds));
  }

  startPlaybackIndicator(duration) {
    if (!this.debugProgressElement) return;
    this.debugProgressElement.style.width = "0%";
    this.debugDuration = duration * 1000;
    this.debugStartTime = Date.now();
    if (this.debugAnimationFrame) {
      cancelAnimationFrame(this.debugAnimationFrame);
    }
    this.updatePlaybackIndicator();
  }

  updatePlaybackIndicator() {
    if (!this.debugProgressElement) return;
    let elapsed = Date.now() - this.debugStartTime;
    let percent = Math.min((elapsed / this.debugDuration) * 100, 100);
    this.debugProgressElement.style.width = percent + "%";
    if (percent < 100) {
      this.debugAnimationFrame = requestAnimationFrame(() => this.updatePlaybackIndicator());
    }
  }

}